import * as types from '../types'
import { taskScoreApi } from 'src/api/TaskScoreApi'
import { taskApi } from 'src/api/TaskApi'
import Vue from 'vue'
import { insertAfter } from '../../utils/utils'

const getDefaultState = () => {
    return {
        loading: true,
        scoreTasks: [],
        filters: {}
    }
}

export default {
    namespaced: true,

    state: getDefaultState(),

    mutations: {
        setLoading (state, loading) {
            state.loading = loading
        },

        setScoreTasks (state, scoreTasks) {
            state.scoreTasks = scoreTasks
        },

        createScoreTask (state, scoreTask) {
            insertAfter(state.scoreTasks, scoreTask, e => e.isRegular)
        },

        updateScoreTask (state, scoreTask) {
            const idx = state.scoreTasks.findIndex(s => s.id === scoreTask.id)
            if (idx >= 0) Vue.set(state.scoreTasks, idx, scoreTask)
        },

        deleteScoreTasks (state, ids) {
            for (const id of ids) {
                const idx = state.scoreTasks.findIndex(t => t.id === id)
                if (idx >= 0) state.scoreTasks.splice(idx, 1)
            }
        },

        /**
         * @param state
         * @param data {TaskProgressEntity}
         */
        updateScoreTaskProgressByEvent (state, data) {
            const task = state.scoreTasks.find(t => t.id === data.taskId)
            if (task) {
                task.execStatus = data.execStatus
                task.progressCurrent = data.progressCurrent
                task.progressTotal = data.progressTotal
                task.waitPosition = data.waitPosition
            }
        },

        setFilters (state, filters) {
            state.filters = filters
        },

        reset: (state) => Object.assign(state, getDefaultState())
    },

    actions: {
        [types.reset]: ({ commit }) => commit('reset'),

        async [types.getScoreTasks] ({ commit, state }, useCache = false) {
            commit('setLoading', true)
            try {
                commit('setScoreTasks', await taskScoreApi.getAll(state.filters, useCache))
            } finally {
                commit('setLoading', false)
            }
        },

        async [types.runTask] ({ dispatch, commit }, { taskId }) {
            await taskApi.runNow(taskId)
        },

        async [types.createScoreTask] ({ commit }, { isRegular, data: scoreTask }) {
            const data = isRegular
                ? await taskScoreApi.createPeriodic(scoreTask)
                : await taskScoreApi.createSimple(scoreTask)

            commit('createScoreTask', data)
        },

        async [types.updateScoreTask] ({ commit }, { isRegular, data: scoreTaskDiff }) {
            const data = isRegular
                ? await taskScoreApi.updatePeriodic(scoreTaskDiff)
                : await taskScoreApi.updateSimple(scoreTaskDiff)

            commit('updateScoreTask', data)
        },

        async [types.deleteScoreTasks] ({ commit }, ids) {
            const data = await taskScoreApi.delete(ids)

            commit('deleteScoreTasks', data.deletes)

            return data
        },

        /**
         * @param commit
         * @param data {TaskProgressEntity}
         * @returns {Promise<void>}
         */
        async [types.updateScoreTaskProgressByEvent] ({ commit }, data) {
            commit('updateScoreTaskProgressByEvent', data)
        },

        /**
         * @param commit
         * @param id {string}
         * @returns {Promise<void>}
         */
        async [types.interruptScoreTask] ({ commit }, id) {
            await taskScoreApi.interrupt(id)
        },

        async [types.applyScoreTasksFilters] ({ commit }, filters = {}) {
            commit('setFilters', filters)
        }
    }
}
