import * as types from '../types'
import { scoresApi } from 'src/api'
import Vue from 'vue'
import { ShowType } from 'src/api/enums/FilterEnums'
import { insertAfter } from '../../utils/utils'

const getDefaultState = () => {
    return {
        scores: [],
        filters: {},
        total: 0,
        loading: true
    }
}

export default {
    namespaced: true,

    state: getDefaultState(),

    getters: {
        scoresById (state) {
            const result = {}
            state.scores.forEach(s => { result[s.id] = s })
            return result
        }
    },

    mutations: {
        setFilters (state, filters = {}) {
            state.filters = filters
        },

        setScores (state, { total, items }) {
            state.total = total
            state.scores = items
        },

        createScore (state, score) {
            insertAfter(state.scores, score, e => e.isPeriodic)
            state.total += 1
        },

        updateScore (state, score) {
            const idx = state.scores.findIndex(s => s.id === score.id)
            if (idx >= 0) Vue.set(state.scores, idx, score)
        },

        deleteScores (state, ids) {
            for (const id of ids) {
                const idx = state.scores.findIndex(m => m.id === id)
                if (idx >= 0) {
                    state.scores.splice(idx, 1)
                    state.total -= 1
                }
            }
        },

        setLoading (state, loading) {
            state.loading = loading
        },

        reset: (state) => Object.assign(state, getDefaultState())
    },

    actions: {
        [types.reset]: ({ commit }) => commit('reset'),

        async [types.applyScoresFilters] ({ commit }, filters = {}) {
            commit('setFilters', filters)
        },

        async [types.getScores] ({ state, commit }, useCache = false) {
            commit('setLoading', true)
            try {
                const scores = await scoresApi.getScores({
                    show: ShowType.ALL,
                    ...state.filters
                }, useCache)

                commit('setScores', scores)
            } finally {
                commit('setLoading', false)
            }
        },

        /**
         * @param commit
         * @param event {TaskContentCreateEntity}
         */
        async [types.createScoreByEvent] ({ commit }, event) {
            commit('createScore', event.content)
        },

        /**
         * @param commit
         * @param score {string}
         * @returns {Promise<void>}
         */
        async [types.updateScore] ({ commit }, score) {
            const newScore = await scoresApi.update(score)
            commit('updateScore', newScore)
        },

        /**
         * @param commit
         * @param ids {string[]}
         * @returns {Promise<string[]>}
         */
        async [types.deleteScores] ({ commit }, { ids, force = false }) {
            const data = await scoresApi.delete(ids, force)

            commit('deleteScores', data.deletes)

            return data
        },

        /**
         * @param commit
         * @param scoreId {string}
         * @returns {Promise<void>}
         */
        async [types.sendScoreToFoms] ({ commit }, scoreId) {
            const newScore = await scoresApi.sendScore(scoreId)
            commit('updateScore', newScore)
        }
    }
}
