import Vue from 'vue'
import Vuex from 'vuex'

import * as types from './types'
import Auth from './modules/auth'
import Users from './modules/users'
import Groups from './modules/group'
import Models from './modules/models'
import ModelTasks from './modules/modelTasks'
import Scores from './modules/scores'
import ScoreTasks from './modules/scoreTasks'
import Reports from './modules/reports'
import FedReports from './modules/fedReports'
import Supervising from './modules/supervising'
import Listener from './modules/listener'
import Logs from './modules/logs'
import NSI from './modules/nsi'
import Notifications from './modules/notifications'
import SyncData from './modules/syncData'
import SyncQueue from './modules/syncQueue'

import { authApi, systemApi } from 'src/api'
import { router } from 'src/router'
import { redirectToLogin } from 'src/utils/utils'

Vue.use(Vuex)

const modules = {
    Auth, Users, Groups, Models, ModelTasks, Scores, ScoreTasks, Reports, FedReports, Supervising, Listener, Logs, NSI, Notifications, SyncData, SyncQueue
}

export default new Vuex.Store({
    strict: process.env.NODE_ENV !== 'production',

    modules,

    state: {
        version: null
    },

    getters: {
        currentUser: state => state.Auth.currentUser,
        isAuthorized: state => !!state.Auth.currentUser,
        isAdmin: (state, { isAuthorized, currentUser }) => isAuthorized ? currentUser.isAdmin : false,
        role: (state, { isAdmin }) => {
            return isAdmin ? 'Администратор' : 'Пользователь'
        },
        userZones: (state, { isAuthorized, currentUser }) => {
            return isAuthorized && currentUser.group ? currentUser.group.zones : {}
        }
    },

    mutations: {
        /**
         * @param state
         * @param version {VersionEntity}
         */
        setVersion (state, version) {
            state.version = version
        }
    },

    actions: {
        /**
         * Сброс всего состояния, приводит store в начальное состояние
         */
        async [types.reset] ({ dispatch }) {
            await Promise.all(
                Object.keys(modules).map(m => dispatch(`${m}/${types.reset}`))
            )
        },

        /**
         * Действие на старте приложения - загрузка данных авторизации,
         *  запрос за текущим пользователем / проверка авторизации в сервисе.
         */
        async [types.startUp] ({ dispatch }) {
            const authData = authApi.getAuth()
            if (!authData) return redirectToLogin(router)

            authApi.setupAxiosAuth(authData.token)

            try {
                await dispatch(`Auth/${types.getCurrentUser}`)

                // Первая партия запросов: узнаем данные о себе, узнаем действует ли авторизация
                await Promise.all([
                    dispatch(`Users/${types.getAllUsers}`),
                    dispatch(types.getVersion),
                    dispatch(`Listener/${types.initListener}`)
                ])
            } catch (e) {
                console.error(e)
                redirectToLogin(router)
            }
        },

        /**
         * Действие при выходе - полный сброс состояния, остановка листнера, выход из сервиса
         * @returns {Promise<*>}
         */
        async [types.tearDown] ({ dispatch }) {
            await dispatch(`Auth/${types.logout}`)
            await dispatch(`Listener/${types.stopListener}`)
            await dispatch(types.reset)
        },

        async [types.getVersion] ({ commit }) {
            const versionEntity = await systemApi.getVersion()
            commit('setVersion', versionEntity)
        }
    }
})
