const locale = 'ru-RU'
const formatters = {
    s: Intl.DateTimeFormat(locale, { month: 'short', year: 'numeric' }),
    d: Intl.DateTimeFormat(locale, { month: 'short', day: '2-digit', year: 'numeric' }),
    dt: Intl.DateTimeFormat(locale, {
        month: 'short',
        day: '2-digit',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit'
    }),
    exact: Intl.DateTimeFormat(locale, {
        month: '2-digit',
        day: '2-digit',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit'
    })
}

/**
 * Фильтр для форматирования дат, как нам удобно
 */
export const dateFilter = (value, preset = 'd', nullText = 'Без даты') => {
    if (value) {
        const date = new Date(value)
        // new Date("что-то") не кидает исключение и не является falsy
        if (date.toString() !== 'Invalid Date') {
            return formatters[preset]
                .format(date)
                .replace(',', '') // убираем запяятую между датой и временем
        }
        return value
    }

    return nullText
}

/**
 * Фильтр для больших чисел, если value не число, пропустит как есть
 * 1123123 -> `1 123 123`
 * -10000  -> -10 000
 */
export const bigNumFilter = (value) => {
    let number = parseFloat(value)

    if (isNaN(number)) {
        return value
    }

    // переведем из копеек в рубли..
    return (number / 100).toLocaleString('ru', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    })
}

/**
 * Фильтр 0..1 -> 0..100%
 * @param value
 * @return {*}
 */
export const percentFilter = (value) => {
    let number = parseFloat(value)

    if (isNaN(number)) {
        return value
    }

    return Math.round(number * 100) + '%'
}

/**
 * Фильтр для округления с точностью до Nх знаков
 * 0.2 + 0.4 = 0.6000000000000001 -> 0.6
 * @param value - значение
 * @param pow - количество знаков
 * @return {string}
 */
export const roundFilter = (value, pow = 2) => {
    let number = parseFloat(value)

    if (isNaN(number)) {
        return value
    }

    return number.toLocaleString('ru', {
        minimumFractionDigits: pow,
        maximumFractionDigits: pow
    })
}

export const ellipsisFilter = (value, len = 25) => {
    if (typeof value === 'string') {
        return value.slice(0, len) + '...'
    }
    return value
}

export const Sizes = {
    BYTE: 'BYTE',
    KBYTE: 'KBYTE',
    MBYTE: 'MBYTE',
    GBYTE: 'GBYTE'
}

const SizesMult = {
    [Sizes.BYTE]: 1,
    [Sizes.KBYTE]: 1024,
    [Sizes.MBYTE]: 1024 * 1024,
    [Sizes.GBYTE]: 1024 * 1024 * 1024
}

const HumanSizes = {
    [Sizes.BYTE]: 'байт',
    [Sizes.KBYTE]: 'Кбайт',
    [Sizes.MBYTE]: 'Мбайт',
    [Sizes.GBYTE]: 'Гбайт'
}

/**
 *
 * @param value {Number}
 * @param income {Sizes}
 * @param outcome {?Sizes}
 * @return {string}
 */
export const sizeFilter = (value, income, outcome) => {
    const bytes = value * SizesMult[income]
    if (!outcome) {
        for (const s of Object.values(Sizes)) {
            if (bytes >= SizesMult[s]) {
                outcome = s
            }
        }
    }
    const res = roundFilter(bytes / SizesMult[outcome], 2)
    const suffix = HumanSizes[outcome]
    return `${res} ${suffix}`
}

/**
 *
 * @param value {string}
 * @return {string}
 */
export const lowercase = (value) => {
    if (typeof value === 'string') return value.toLowerCase()

    return value
}

export const spaceAfterComma = value => value.split(',').join(', ')
